//===============================================================================
// Microsoft patterns & practices Enterprise Library
// Configuration Application Block
//===============================================================================
// Copyright  Microsoft Corporation.  All rights reserved.
// THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY
// OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT
// LIMITED TO THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
// FITNESS FOR A PARTICULAR PURPOSE.
//===============================================================================

using System.Collections.Generic;

namespace Microsoft.Practices.EnterpriseLibrary.Configuration
{
	/// <summary>
	/// Represents a configuration source that is backed by a dictionary of named objects.
	/// </summary>
	public class DictionaryConfigurationSource : IConfigurationSource
	{
		private Dictionary<string, object> sections;

		/// <summary>
		/// Initialize a new instance of the <see cref="DictionaryConfigurationSource"/> class.
		/// </summary>
		public DictionaryConfigurationSource()
		{
			this.sections = new Dictionary<string,object>();
		}

		/// <summary>
		/// Retrieves a specified configuration section from configuration.
		/// </summary>
		/// <param name="sectionName">The section name to retrieve.</param>
		/// <returns>The configuration for the section.</returns>
		public object GetSection(string sectionName)
		{
			// we need to do this because we want to mimic System.Configuration
			// and return null if not found
			if (sections.ContainsKey(sectionName))
			{
				return sections[sectionName];
			}
			return null;
		}

		/// <summary>
		/// <para>Add a named section to the source.</para>
		/// </summary>
		/// <param name="name">The name of the section to add.</param>
		/// <param name="section">The section object to add.</param>
		public void Add(string name, object section)
		{
			sections.Add(name, section);
		}

		/// <summary>
		/// Removes a named section from the source.
		/// </summary>
		/// <param name="name">The name of the section to remove.</param>
		/// <returns><see langword="true"/> if the section was removed; otherwise, <see langword="false"/>.</returns>
		public bool Remove(string name)
		{
			return sections.Remove(name);
		}		

		/// <summary>
		/// Determines if a section name exists in the source.
		/// </summary>
		/// <param name="name">The section name to find.</param>
		/// <returns><see langword="true"/> if the section exists; otherwise, <see langword="false"/></returns>
		public bool Contains(string name)
		{
			return sections.ContainsKey(name);
		}
	}
}
